//
//  TencentCaptcha.h
//  TencentCaptcha
//
//  Created by Xu Sensheng on 2024-10-31.
//  Copyright © 2024 Tecent Inc. All rights reserved.
//

#import <Foundation/Foundation.h>



#if DEBUG
#define AVAILABLE_ONLY_IF_DEBUG
#else
#define AVAILABLE_ONLY_IF_DEBUG NS_UNAVAILABLE
#endif

#pragma mark -

@class WKWebView;

/// 验证码服务配置
/// - 用于在 `+[TencentCaptcha setupWithConfigurationHandler:]` 中配置服务
/// - SeeAlso:
///     - Objective C: `+[TencentCaptcha setupWithConfigurationHandler:]`,
///     - Swift: `TencentCaptcha.setup(configurationHandler:)`
@interface TencentCaptchaConfiguration : NSObject

- (nonnull instancetype)init NS_UNAVAILABLE;

@property (nonnull, nonatomic, copy) NSString *appID;


/// Tars服务器地址
/// - Note: 如果不清楚用途，请不要设置
@property (nullable, nonatomic, copy) NSURL *tarsServerURL;

/// 设置验证码页面的Host
/// - Note:
///     - 只能设置URL中的host部分，例如， `https://host/path/to/html` 中的host部分，其它部分目前不支持修改
///     - 如果host不合法，导致无法生成正确的页面URL，则该设置无效
///     - 设置为nil可以使其保持为默认值
///     - 如果不清楚用途，请不要设置
@property (nullable, nonatomic, copy) NSString *captchaPageHost;

/// 验证码html页面的地址
/// - Note:
///     - 这只是一个用于调试的选项，正式版本中无效。
///     - 如果不清楚用途，请不要设置。
/// - Warning:
///     - 这只是一个用于调试的选项，会与`captchaPageHost`冲突。
///     - 如果两者均有值，最终页面的URL会先以`webPageURL`为基础，然后将host部分替换为`captchaPageHost`。
@property (nullable, nonatomic, copy) NSURL *webPageURL AVAILABLE_ONLY_IF_DEBUG;

/// 是否允许使用缓存的token以加速请求过程
/// - Note: 默认为YES，缓存策略能满足绝大多数场景的需求。如果您特别关注实时风险，请设置为NO
@property (nonatomic, assign) BOOL usesCacheToken;

/// 风险识别业务的渠道号
/// - Note:
///     - 默认为0，此时关闭TuringShield风险识别对外接口
///     - 只有设置非0值时，TuringShield风险识别对外接口才生效
///     - 如果不清楚用途，请不要设置。
/// - SeeAlso: 风险识别对外接口类 `MiniTuringShield`
@property (nonatomic, assign) uint32_t miniTuringShieldChannel;

/// 验证码请求的超时时长，单位秒
/// - Note: 默认为60秒
///     - 验证码加载包含三个阶段：本地设备风险检测、本地设备风险检测数据采集、H5页面加载
///     - 超时时长为三个阶段的总和，内部会跟踪并且动态调整各个阶段的超时时长控制
@property (nonatomic, assign) NSTimeInterval timeout;

/// 本地设备风险检测的请求超时时长，单位秒
@property (nonatomic, assign) NSTimeInterval timeoutForNativeRequest
__attribute__((deprecated("NOT WORK! Use 'timeout' instead.")));

/// 本地设备风险检测的采集超时时长，单位秒
@property (nonatomic, assign) NSTimeInterval timeoutForNativePacking
__attribute__((deprecated("NOT WORK! Use 'timeout' instead.")));

/// 页面加载的超时时长，单位秒
@property (nonatomic, assign) NSTimeInterval timeoutForWebViewLoading
__attribute__((deprecated("NOT WORK! Use 'timeout' instead.")));

@end

/// 深夜模式选项
typedef enum : NSUInteger {
    /// 默认
    TencentCaptchaOptionsDarkModeDefault,
    /// 随系统开启
    TencentCaptchaOptionsDarkModeTrue,
    /// 总是关闭
    TencentCaptchaOptionsDarkModeFalse,
    /// 强制开启
    TencentCaptchaOptionsDarkModeForce,
} TencentCaptchaOptionsDarkMode;

/// 开关选项
typedef enum : NSUInteger {
    TencentCaptchaOptionsSwitchDefault,
    TencentCaptchaOptionsSwitchOn,
    TencentCaptchaOptionsSwitchOff,
} TencentCaptchaOptionsSwitch;


/// 验证码请求参数
@interface TencentCaptchaOptions : NSObject

/// 自定义透传参数，业务可用该字段传递少量数据，该字段的内容会被带入 callback 回调的对象中。
@property (nonatomic, strong, nullable) NSString *bizState;

/// 深夜模式
@property (nonatomic, assign) TencentCaptchaOptionsDarkMode darkModeEnabled;

/// 帮助按钮开关
@property (nonatomic, assign) TencentCaptchaOptionsSwitch needsFeedback;

/// 自定义帮助按钮的链接
/// - Note: 如`customFeedbackURL`非空，则`needsFeedback`选项无效，帮助按钮总是显示，且点击时跳转到该URL
@property (nonatomic, strong, nullable) NSString *customFeedbackURL;

/// 指定验证码提示文案的语言，优先级高于控制台配置
@property (nonatomic, strong, nullable) NSString *userLanguage;

/// 适老化模式
@property (nonatomic, assign) TencentCaptchaOptionsSwitch agedEnabled;

/// 定义验证码展示方式
/// - Remark: 支持的值：
///     - popup（默认）弹出式，以浮层形式居中弹出展示验证码
///     - embed 嵌入式，以嵌入指定容器元素中的方式展示验证码。
@property (nonatomic, strong, nullable) NSString *type;

/// 是否在验证码加载过程中显示loading框
@property (nonatomic, assign) TencentCaptchaOptionsSwitch loadingEnabled;

/// CaptchaAppId 加密校验串
/// @discussion 详见 https://www.tencentcloud.com/zh/document/product/1159/49680?lang=zh&pg=#captchaAppid
@property (nonatomic, copy, nullable) NSString *encryptedAppID;




/// 设置额外的参数
/// - Parameters:
///   - value: 参数的值
///   - name: 参数的关键字名称
/// - Note:
///   - 当name对应的参数不存在，会插入新值，已存在时，新值会覆盖旧值；当value为nil时，将删除该参数
///   - 参数将传递至验证码的HTML部分。如需了解关键字名称的详情，请参阅验证码HTML的文档
- (void)setExtractParameter:(nullable id)value forName:(nonnull NSString *)name;

/// 当前已设置的额外参数
@property (nonatomic, strong, readonly, nonnull) NSDictionary<NSString *, id> *extractParameters;

/// 参数和中间过程的回调
/// - Note: 当您需要接收中间参数时，请设置此回调
@property (nonatomic, copy, nullable) void(^optionsCallback)(NSString *_Nonnull functionName, NSDictionary<NSString *, id> *_Nonnull data);

@end


@class MiniTuringShield;

/// 验证码服务的实现类
/// - 提供主要的验证码服务能力
/// - Example:
/// ```swift
/// // Step 1. 引导用户同意用户隐私协议，然后设置`userDidAcceptPrivacyPolicy`
/// TencentCaptcha.userDidAcceptPrivacyPolicy = true
/// // Step 2. 初始化SDK
/// TencentCaptcha.setup(...)
/// // Step 3. 生成WebView并配置
/// webView = WKWebView(...)
/// self.view.addSubview(webView)
/// // Step 4. 生成和展示验证码
/// TencentCaptcha.requestCaptcha(...)
///
@interface TencentCaptcha : NSObject

- (nonnull instancetype)init NS_UNAVAILABLE;

/// 验证码服务的单例
+ (nonnull instancetype)sharedService NS_SWIFT_NAME(shared());

/// 用户是否已同意隐私政策
/// 调用任何SDK功能前，请先引导用户同意隐私协议，然后先将此类属性设为 `YES` / `true`
/// - Example:
/// ```swift
///     // 用户同意隐私协议后
///     TencentCaptcha.userDidAcceptPrivacyPolicy = true
/// ```
/// - Warning: 用户未同意隐私协议前，SDK不提供验证码服务
@property (nonatomic, assign, class) BOOL userDidAcceptPrivacyPolicy;

@property (nonatomic, readonly, nonnull, class) TencentCaptchaConfiguration *configuration;

/// 配置服务选项
/// - Parameters:
///     - configurationHandler: 如需修改配置，请修改此回调函数中提供的configuration中的属性
///
/// - Example: 最小化设置
/// ```
/// TencentCaptcha.setup { config in
///     config.appID = __YOUR_CAPTCHA_APP_ID__ // 请使用您的验证码App ID替换__YOUR_CAPTCHA_APP_ID__
/// }
/// ```
/// - Note: 请在尽可能靠前的时机配置服务，并确保在使用服务前配置
/// - SeeAlso: `TencentCaptchaConfiguration`
+ (void)setupWithConfigurationHandler:(nonnull void(^)(TencentCaptchaConfiguration *_Nonnull))configurationHandler NS_SWIFT_NAME(setup(configurationHandler:));

/// 请求验证码
/// - Parameters:
///   - webView: 用于展示验证码的网页视图，需要由调用方初始化
///   - options: 验证码请求参数
///   - completionHandler: 验证结束的回调
///
/// - Example:
/// ```swift
/// TencentCaptcha.shared().requestCaptcha(withWebView: webView, options: options) { error, result in
///     if let result = result {
///         // 当token不为空时，验证码验证成功
///     }
///     else {
///         // 否则error会输出错误，具体错误域和错误号请参见文档
///     }
/// }
/// ```
///
/// - Warning: 由于某些系统限制，SDK暂时无法接收来自StoryBoard生成的WKWebView。请使用代码方式初始化一个WKWeView并传递给SDK，例如：
/// ```swfit
///     let contentController = WKUserContentController()
///     let config = WKWebViewConfiguration()
///     config.userContentController = contentController
///     webView = WKWebView(frame: self.view.bounds, configuration: config)
///     webView.autoresizingMask = [.flexibleWidth, .flexibleHeight]
///     self.view.addSubview(webView)
/// ```
- (void)requestCaptchaWithWebView:(nonnull WKWebView *)webView options:(nullable TencentCaptchaOptions *)options withCompletionHandler:(nonnull void(^)(NSError *_Nullable error, NSDictionary *_Nullable result))completionHandler NS_SWIFT_NAME(requestCaptcha(withWebView:options:withCompletionHandler:));

@end


/// 图灵盾设备风险实现类
/// - Example:
/// ```swift
/// // Step 1. 在你的验证码初始化过程(`TencentCaptcha.setup`）中追加渠道号设置
/// TencentCaptcha.setup { config in
/// ...
/// RootViewController.RCEChannel = __YOUR_CHANNEL_ID__ // 请联系我们获得你的渠道号并配置到此处
/// ...
/// }
/// // Step 2. 获得风险token，参见`fetchUniversalRiskMessage`
/// MiniTuringShield.standardService()?.fetchUniversalRiskMessage(...)
/// ```
/// - Important:
/// 在没有设置好 `TencentCaptchaConfiguration.miniTuringShieldChannel` 前，无法使用该类提供的服务。如您需要使用图灵盾风险识别能力，请与我们联系并获取渠道号
@interface MiniTuringShield : NSObject

- (nonnull instancetype)init NS_UNAVAILABLE;

/// 服务的单例
+ (nullable instancetype)standardService;

/// 请求风险识别
/// - Parameters:
///   - extractAPIChecking: 保留接口，用于日后扩展，请保持为nil
///   - shouldUseCachedMessage: 是否允许使用缓存的token以加速请求过程，如无特殊需求，请设置为YES
///   - includesOutdatedMessage: 是否允许在请求失败时，返回缓存，如无特殊需求，请设置为YES
///   - completionHandler: 请求完成后的回调，如成功，error为空，message为风险token，如失败，error为错误原因，message为空
///
/// - Example:
/// ```swift
/// MiniTuringShield.standardService()?.fetchUniversalRiskMessage(withExtractAPIChecking: nil, usingCachedMessage: trincludesOutdatedMessage: true) { token, error in
///     if let token = token {
///         // 当token不为空时，获取成功
///     }
///     else {
///         // 否则error会输出错误，具体错误域和错误号请参见文档
///     }
/// }
/// ```
///
/// - Important:
///   - 请求总是异步返回，返回的线程非固定，请注意线程安全；
///   - 当你需要得到错误原因时，请同时收集error.domain和error.code，否则错误信息可能不完整。
- (void)fetchUniversalRiskMessageWithExtractAPIChecking:(nullable NSArray<NSString *> *)extractAPIChecking
                                     usingCachedMessage:(BOOL)shouldUseCachedMessage
                                includesOutdatedMessage:(BOOL)includesOutdatedMessage
                                  withCompletionHandler:(nonnull void (^)(NSString *_Nullable message, NSError *_Nullable error))completionHandler
    NS_SWIFT_NAME(fetchUniversalRiskMessage(withExtractAPIChecking:usingCachedMessage:includesOutdatedMessage:completionHandler:));


/// 激活验签
/// - Parameter token: 验签票据
/// - Parameter error: 返回的错误信息
/// - Returns 成功时返回YES，否则返回NO
/// - Throws: 当失败时，将返回错误，错误类型请参阅文档
- (BOOL)activateSignatureToken:(nonnull NSString *)token error:(NSError *__autoreleasing _Nullable *_Nullable)error;

/// 请求签名
/// - Parameter data: 被签名的数据块
/// - Parameter error: 返回的错误信息
/// - Returns 成功时返回签名信息，否则返回空
/// - Throws: 当失败时，将返回错误，错误类型请参阅文档
- (nullable NSString *)getSignatureForData:(nonnull NSData *)data error:(NSError *__autoreleasing _Nullable *_Nullable)error;


@end
